<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class ShopifyService
{
    public function hasOnlineStoreChannel($shopDomain, $accessToken)
    {
        // Heredoc syntax for GraphQL query (Make sure no extra spaces)
        $graphqlQuery = <<<GRAPHQL
        {
            channels {
                edges {
                    node {
                        id
                        name
                        handle
                    }
                }
            }
        }
GRAPHQL;

        // Make sure the URL is formatted correctly.
        $url = "https://{$shopDomain}/admin/api/2024-04/graphql.json"; // Shopify API version

        // Sending POST request to Shopify GraphQL API
        $response = Http::withHeaders([
            'X-Shopify-Access-Token' => $accessToken,
            'Content-Type' => 'application/json',
        ])->post($url, [
            'query' => $graphqlQuery
        ]);

        // Check for failed request
        if ($response->failed()) {
            Log::error('Shopify GraphQL API request failed', [
                'shop' => $shopDomain,
                'status' => $response->status(),
                'response' => $response->body(),
            ]);
            return false;  // Return false if API request fails
        }

        // Process response JSON
        $json = $response->json();
        // Safely access the 'edges' array within the response
        if (isset($json['data']['channels']['edges'])) {
            $channels = $json['data']['channels']['edges'];
        } else {
            $channels = [];
        }

        // Debugging: Log the channels fetched from Shopify
        Log::debug('Fetched Shopify sales channels', [
            'shop' => $shopDomain,
            'channels' => $channels,
        ]);

        // Check if any channel has the 'online-store' handle
        foreach ($channels as $channel) {
            if (isset($channel['node']['handle']) && $channel['node']['handle'] === 'online-store') {
                return true;  // Return true if online store is enabled
            }
        }

        return false;  // Return false if 'online-store' channel is not found
    }
}
